/** @file
 * @brief MAGResWCH DLL API for controlling the MAGRes board.
 *
 * */

#pragma once

#ifdef MAGRESWCH_EXPORTS
#define MAGRESWCH_API __declspec(dllexport)
#else
#define MAGRESWCH_API __declspec(dllimport)
#endif


#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

/** @brief The board handle */
typedef struct dio_handle_s *dio_handle_h;

/** @brief The handle for a bmp file */
typedef struct dio_bmp_s *dio_bmp_h;

/** @brief Errors represented by lpnStatus */
typedef enum {
  /** @brief Cannot allocate lpnMasterHandle. */
  ERR_CANNOT_ALLOCATE_LPNMASTERHANDLE = -1,
  /** @brief Invalid CH367 handle returned. */
  ERR_INVALID_CH367_HANDLE = -2,
  /** @brief Cannot get io_reg address for the CH367. */
  ERR_CANNOT_GET_IO_REG_ADDR = -3,
  /** @brief Commanded step size is less than or equal to zero. */
  ERR_STEP_SIZE_LESS_EQUAL_ZERO = -4,
  /** @brief Number of bytes to write is larger than available memory. */
  ERR_WRITE_LONGER_THAN_AVAIL_MEM = -5,
  /** @brief Number of bytes to write is zero, needs to be at least 1. */
  ERR_WRITE_ZERO_BYTES = -6,
  /** @brief Number of bytes to read is specified to be zero, needs to be at least 1. */
  ERR_READ_ZERO_BYTES = -7,
  /** @brief Cannot read SPI. */
  ERR_SPI_READ_ERR = -8,
  /** @brief Cannot stream SPI. */
  ERR_SPI_STREAM_ERR = -9,
  /** @brief Cannot write to the SPI. */
  ERR_SPI_WRITE_ERR = -10,
  /** @brief Tried to access an invalid SPI address. */
  ERR_SPI_ADDR_OUT_OF_BOUNDS = -11,
  /** @brief Tried to open invalid BMP file. */
  ERR_BMP_INVALID_FILE = -12
} dio_errors_e;

/** @brief External Trigger modes */
typedef enum {
  /** @brief Ignore the external trigger line */
  DIO_EXT_TRIG_IGNORE = 0,
  /** @brief Trigger and change state on the rising edge */
  DIO_EXT_TRIG_RISING_EDGE,
  /** @brief Trigger and change state on the falling edge */
  DIO_EXT_TRIG_FALLING_EDGE,
  /** @brief Trigger and change state when the line is held high */
  DIO_EXT_TRIG_HIGH_HOLD,
  /** @brief Trigger and change state when the line is held low */
  DIO_EXT_TRIG_LOW_HOLD
} dio_ext_trigger_modes_e;


typedef enum {
  DIO_ADRSR_PIN_LOW = 0,
  DIO_ADRSR_PIN_HIGH
} dio_adrsr_pin_state_e;

/** @brief Pixel blending modes when using the OLED screen buffer */
typedef enum {
  /** @brief Overwrites what's there. */
  DIO_OLED_MODE_OVERWRITE = 0,
  /** @brief Or current pixel with new pixel value. */
  DIO_OLED_MODE_OR,
  /** @brief Overwrite current pixel if new pixel is greater than existing pixel. */
  DIO_OLED_MODE_GREATER_THAN,
  /** @brief Xor current pixel with new pixel value. */
  DIO_OLED_MODE_XOR,
  /** @brief Overwrite current pixel with new pixel value as long as the new value is nonzero. */
  DIO_OLED_MODE_NONZERO
} dio_oled_write_mode_e;

/*
typedef enum {
  DIO_OLED_BUFF_SEND_MODE_FULL = 0, //sends the full buffer
  DIO_OLED_BUFF_SEND_MODE_DIRTY, //sends the 'dirty area' of the buffer
  DIO_OLED_BUFF_SEND_MODE_QUAD_1 //divides into quads and sends dirty quad
} dio_oled_buff_send_mode;
*/

/**
 * @brief Provides a one time initialization of a master board and all slave
 * boards associated with it.
 *
 * Returns the handle to the board.
 *
 * Initialize the DIO system configuration with one command.
 *
 * @param nMasterNum      number of the master board in the sytem.
 * @param lpnMasterHandle Returned master handle, 0 or NULL on error.
 * @param lpnStatus       returned status, 0 on success, < 0 failure.
 *
 */
MAGRESWCH_API void DioInitialize(short nMasterNum, dio_handle_h *lpnMasterHandle, int16_t* lpnStatus);

/**
* @brief Changes the board state from Halt to Pause.
*
* The board must be in HALT state, otherwise it does nothing.
*
* @param nMasterHandle  master board handle
* @param lpnStatus      return status, 0 on success, <0 on failure
*/
MAGRESWCH_API void DioArm(dio_handle_h nMasterHandle, int16_t* lpnStatus);

/**
 * @brief Changes specified master board state to HALT
*
* Change board program counter value to zero. Calling Dio Arm then DioTrig after
* the board is in a halt state will cause the board to run from the beginning.
*
* @param nMasterHandle  master board handle
* @param lpnStatus      returned status, 0 on success < 0 on failure
*/
MAGRESWCH_API void DioHalt(dio_handle_h nMasterHandle, int16_t* lpnStatus);

/**
 * @brief Changes the specified master board state from a RUN to PAUSE state
 *
 * Board must be in a RUN state, otherwise the function does nothing.
 *
 * @param nMasterHandle master board handle
 * @param lpnStatus     returned status, 0 on success, < 0 failure.
 *
 */
MAGRESWCH_API void DioPause (dio_handle_h nMasterHandle, int16_t* lpnStatus);


/**
 * @brief Takes the specified master board state from PAUSE to RUN
 *
 * @param nMasterHandle master board handle
 * @param lpnStatus     returned status, 0 on success, < 0 failure.
 */
MAGRESWCH_API void DioTrig(dio_handle_h nMasterHandle,
    int16_t* lpnStatus);

/**
 * @brief Reads a block of control memory from the specified
 * board
 *
 * Board must be in a HALT state before calling these functions.
 *
 * @param nHandle     master board handle
 * @param lnpdwVector Array of double words up to 16K elements
 * @param dwStart     Starting address to read
 * @param dwSize      Number of steps to read
 * @param lpnStatus   returned status, 0 on success, < 0 failure, > 0 number of
 * bytes read
 *
 */
MAGRESWCH_API void DioReadCtrlMemory (dio_handle_h nHandle,
    uint32_t* lnpdwVector,
    uint32_t dwStart,
    uint32_t dwSize,
    int16_t* lpnStatus);


/**
 * @brief Reads a block of output memory from the specified
 * board
 *
 * Board must be in a HALT state before calling these functions.
 *
 * @param nHandle     master board handle
 * @param lnpdwVector Array of double words up to 16K elements
 * @param dwStart     Starting address to read
 * @param dwSize      Number of steps to read
 * @param lpnStatus   returned status, 0 on success, < 0 failure, > 0 number of
 * bytes read
 *
 */
MAGRESWCH_API void DioReadOutMemory (dio_handle_h nHandle,
    uint32_t* lnpdwVector,
    uint32_t dwStart,
    uint32_t dwSize,
    int16_t* lpnStatus);


/**
 * @brief Writes a block of data to the control memory of the
 * specified board.
 *
 * Board must be in a HALT state before calling these functions.
 *
 * @param nMasterHandle master board handle
 * @param lpdwMemory    array of double words
 * @param dwStart       starting address to write
 * @param dwSize        number of steps to write.
 * @param lpnStatus     returned status, 0 on success, < 0 failure.
 */
MAGRESWCH_API void DioWriteCtrlMemory(dio_handle_h nHandle,
    uint32_t* lpdwMemory,
    uint32_t dwStart,
    uint32_t dwSize,
    int16_t* lpnStatus);

/**
 * @brief Writes a block of data to the output memory of the
 * specified board.
 *
 * Board must be in a HALT state before calling these functions.
 *
 * @param nMasterHandle master board handle
 * @param lpdwMemory    array of double words
 * @param dwStart       starting address to write
 * @param dwSize        number of steps to write.
 * @param lpnStatus     returned status, 0 on success, < 0 failure.
 */
MAGRESWCH_API void DioWriteOutMemory(dio_handle_h nHandle,
    uint32_t* lpdwMemory,
    uint32_t dwStart,
    uint32_t dwSize,
    int16_t* lpnStatus);

/**
 * @brief Returns the specified master board program counter value.
 *
 * Represents the address of the current step being executed.
 *
 * @param nHandle     master board handle
 * @param lpdwCounter returned program counter value
 * @param lpnStatus   returned status, 0 on success, < 0 failure, > 0 num bytes
 * read
 */
MAGRESWCH_API void DioReadProgramCounter(dio_handle_h nHandle,
    int32_t* lpdwCounter,
    int16_t* lpnStatus);

/**
 * @brief Writes the specified master board program counter value.
 *
 * The board must be in pause state for this function to be successful.
 *
 * @param nHandle     master board handle
 * @param dwCounter   new program counter value
 * @param lpnStatus   returned status, 0 on success, < 0 failure, > 0 num bytes
 * read
 */
MAGRESWCH_API void DioWriteProgramCounter(dio_handle_h nHandle,
    int32_t dwCounter,
    int16_t* lpnStatus);

/**
 * @brief Returns the specified master board current output value
 *
 * @param nHandle     master board handle
 * @param lpdwCounter returned program counter value
 * @param lpnStatus   returned status, 0 on success, < 0 failure, > 0 num bytes
 * read
 */
MAGRESWCH_API void DioReadCurrentIO(dio_handle_h nHandle,
    int32_t* lpdwCounter,
    int16_t* lpnStatus);

/**
 * @brief Writes the specified master board current output value.
 *
 * The board must be in pause.
 *
 * @param nHandle     master board handle
 * @param dwCounter   new program counter value
 * @param lpnStatus   returned status, 0 on success, < 0 failure, > 0 num bytes
 * read
 */
MAGRESWCH_API void DioWriteCurrentIO(dio_handle_h nHandle,
    int32_t dwCounter,
    int16_t* lpnStatus);



/**
 * @brief Reads the status register from the specified board
 *
 * | bits | id | description |
 * | --- | --- | --- |
 * | 15-5   | R | Reserved
 * | 4-2     | STATE | board state: <br> 000 HALT <br> 011 Pause <br> 1xx Run |
 * | 1-0    | UNUSED | |
 *
 * @param nHandle     master board handle
 * @param lpwData     returned status register value
 * @param lpnStatus   returned status, 0 on success, < 0 failure.
 */
MAGRESWCH_API void DioReadStatusRegister(dio_handle_h nHandle,
    int16_t* lpwData,
    int16_t* lpnStatus);


/*
 * @brief Reads/writes x register from/to the specified board
 *
 * called to simulate external event lines value.
 *
 * @param nHandle     master board handle
 * @param lpwData     returned register value
 * @param wData       value to be written
 * @param lpnStatus   returned status, 0 on success, < 0 failure.
 */
/*
MAGRESWCH_API void DioReadXRegister(dio_handle_h nHandle,
    int32_t* lpwData,
    int16_t* lpnStatus);

MAGRESWCH_API void DioWriteXRegister(dio_handle_h nHandle,
    int32_t* wData,
    int16_t* lpnStatus);
    */


/**
 * @brief Changes state of specified master board to HALT.
 *
 * @param nHandle     master board handle
 * @param lpnStatus   returned status, 0 on success, < 0 failure.
 */
MAGRESWCH_API void DioReset(dio_handle_h nHandle,
    int16_t* lpnStatus);

/*
 * @brief Sets up the specified master board external event lines trigger mode
 *
 * DioTrig function or the external trigger control line can trigger the board
 * regardless of the trigger mode set by this function. Board must be in a PAUSE
 * state to enable triggering.
 *
 * @param nHandle     master board handle
 * @param nMode       specifies one of several trigger modes
 * @param lpnStatus   returned status, 0 on success, < 0 failure.
 */
/*
MAGRESWCH_API void DioSetupTriggerMode(dio_handle_h nHandle,
    int16_t nMode,
    int16_t* lpnStatus);
    */


/*
 * @brief Sets up the specified master board external event lines source.
 *
 *
 * @param nMasterHandle master board handle
 * @param nSource       0 source is external event lines/1 X register
 * @param lpnStatus     returned status, 0 on success, < 0 failure.
 */
/*
MAGRESWCH_API void DioSetupTriggerXEventSource(dio_handle_h nMasterHandle,
    int16_t nSource,
    int16_t* lpnStatus);
    */

/**
 * @brief Executes the specified master board program for a number of steps
 *
 * Board must be in a PAUSE state.
 *
 * Each step is executed by changing the board state from PAUSE to RUN state and
 * then to PAUSE state again.
 *
 * @param nMasterHandle master board handle
 * @param dwSteps       Number of steps to execute
 * @param lpnStatus     returned status, 0 on success, < 0 failure.
 */
MAGRESWCH_API void DioStep(dio_handle_h nMasterHandle,
    int32_t dwSteps,
    int16_t* lpnStatus);

/**
 * @brief Returns the specified master board max data lines available to program
 *
 * @param nHandle     master board handle
 * @param lpdwCounter returned data lines value
 * @param lpnStatus   returned status, 0 on success, < 0 failure, > 0 num bytes
 * read
 */
MAGRESWCH_API void DioReadMaxDataLines(dio_handle_h nHandle,
    int32_t* lpdwCounter,
    int16_t* lpnStatus);


/**
 * @brief Clears the control memory on the board
*
* @param nMasterHandle  master board handle
* @param lpnStatus      returned status, 0 on success < 0 on failure
*/
MAGRESWCH_API void DioClearCtrlMemory(dio_handle_h nMasterHandle, int16_t* lpnStatus);

/**
 * @brief Clears the output memory on the board
*
* @param nMasterHandle  master board handle
* @param lpnStatus      returned status, 0 on success < 0 on failure
*/
MAGRESWCH_API void DioClearOutMemory(dio_handle_h nMasterHandle, int16_t* lpnStatus);


/**
 * @brief Streams the SPI.
 *
 * Writes the contents of `write_buff` to the SPI peripheral at the given
 * address, and read any incoming data that is coming in simultaneously to fill
 * the `read_buff`.
 *
 * Address 0 is the FPGA, and it is recommended to use the DIO API to
 * communicate with it.
 *
 * @param nMasterHandle  master board handle
 * @param addr           address of the SPI peripheral 0-7.
 * @param write_buff     pointer to byte-array of what to write
 * @param read_buff      pointer to byte-array of what to read
 * @param len            length of both the read and write buffer
 * @param lpnStatus      returned status, 0 on success < 0 on failure
*/
MAGRESWCH_API void DioStreamSPI(dio_handle_h nMasterHandle,
    uint8_t addr,
    uint8_t *write_buff,
    uint8_t *read_buff,
    size_t len,
    int16_t* lpnStatus);

/**
 * @brief Writes to the SPI
 *
 * Writes the `write_buff` to the SPI peripheral at the
 * address, and ignore any any incoming data
 *
 * Address 0 is the FPGA, and it is recommended to use the DIO API to
 * communicate with it.
 *
 * @param nMasterHandle  master board handle
 * @param addr           address of the SPI peripheral 0-7.
 * @param write_buff     pointer to byte-array of what to write
 * @param len            length of both the read and write buffer
 * @param lpnStatus      returned status, 0 on success < 0 on failure
*/
MAGRESWCH_API void DioWriteSPI(dio_handle_h nMasterHandle,
    uint8_t addr,
    uint8_t *write_buff,
    size_t len,
    int16_t* lpnStatus);

/**
 * @brief Reads the SPI
 *
 * Accomplishes this by writing zeroes the SPI peripheral at the
 * address, and fill `read_buff` with any incoming data
 *
 * Address 0 is the FPGA, and it is recommended to use the DIO API to
 * communicate with it.
 *
 * @param nMasterHandle  master board handle
 * @param addr           address of the SPI peripheral 0-7.
 * @param write_buff     pointer to byte-array of what to write
 * @param read_buff      pointer to byte-array of what to read
 * @param len            length of both the read and write buffer
 * @param lpnStatus      returned status, 0 on success < 0 on failure
*/
MAGRESWCH_API void DioReadSPI(dio_handle_h nMasterHandle,
    uint8_t addr,
    uint8_t *read_buff,
    size_t len,
    int16_t* lpnStatus);

/**
 * @brief Resets and initializes the DDS. Can only be sent during HALT.
*
* @param nMasterHandle  master board handle
* @param lpnStatus      returned status, 0 on success < 0 on failure
*/
MAGRESWCH_API void DioResetDDS(dio_handle_h nMasterHandle, int16_t* lpnStatus);

/**
 * @brief Sets the external trigger mode during the PAUSE and RUN states.
 *
 * The board can be triggered from an external line, but defaults to ignoring
 * any changes on this external line. This function sets the action of the board
 * in either the PAUSE or the RUN state. The different functionality is described in
 * `dio_ext_trigger_modes_e`
 *
 * The PAUSE state will transition to the RUN state depending on the value
 * sent in `nPauseMode`. The RUN state will transition to the PAUSE state
 * depending on the value sent in `nRunMode`.
*
* @param nMasterHandle  master board handle
* @param nPauseMode     trigger mode during the pause state
* @param nRunMode       trigger mode during the run state
* @param lpnStatus      returned status, 0 on success < 0 on failure
*/
MAGRESWCH_API void DioSetupTriggerMode(dio_handle_h nMasterHandle,
    uint8_t nPauseMode,
    uint8_t nRunMode,
    int16_t* lpnstatus);

/**
 * @brief Closes the connection to the board, and releases the handle from being
 * used.
 *
 * Passing a handle of 0 or NULL to DioClose will perform no action.
*
* @param nMasterHandle  master board handle
* @param lpnStatus      returned status, 0 on success < 0 on failure
*/
MAGRESWCH_API void DioClose(dio_handle_h nMasterHandle, int16_t* lpnStatus);

MAGRESWCH_API void DioSetADRSR_Low(dio_handle_h nMasterHandle, uint8_t io_byte);
MAGRESWCH_API void DioSetADRSR_High(dio_handle_h nMasterHandle, uint8_t io_byte);

/**
 * @brief Sets up and turns on the OLED.
 *
 * Must be called before any function manipulating the screen is called.
*
* @param nMasterHandle  master board handle
* @param lpnStatus      returned status, 0 on success < 0 on failure
*/
MAGRESWCH_API void DioSetupOLED(dio_handle_h nMasterHandle, int16_t* lpnStatus);


/**
 * @brief Writes a value to a pair of pixels immediately.
 *
 * The x position of the pixel pair is a column, and goes from 0-127, since
 * every column is two pixels wide. Pixel rows are indexed as follows: zero being
 * the top of the screen, 63 being the lowest part of the screen
 *
 * @param nMasterHandle  master board handle
 * @param column         pixel column
 * @param y              pixel row
 * @param val            pixel pair value, the upper nibble being the leftmost
 * pixel
*/
MAGRESWCH_API void DioWriteOLEDPixelPairImm(dio_handle_h nMasterHandle, uint8_t column, uint8_t y, uint8_t val);

/**
 * @brief Writes a value to a pair of pixels in the local screen buffer.
 *
 * This is not written to the screen until the buffer is sent to the device. The
 * x position of the pixel pair is a column, and goes from 0-127, since every
 * column is two pixels wide. Pixel rows are indexed as follows: zero being the top
 * of the screen, 63 being the lowest part of the screen.
 *
 * @param nMasterHandle  master board handle
 * @param mode           buffer overwrite mode
 * @param column         pixel column
 * @param y              pixel row
 * @param val            pixel pair value, the upper nibble being the leftmost
 * pixel
*/
MAGRESWCH_API void DioWriteOLEDPixelPairBuff(dio_handle_h nMasterHandle, dio_oled_write_mode_e mode,
    uint8_t column, uint8_t y, uint8_t val);

/**
 * @brief Writes a pixel to the screen at the designated coordinate immediately.
 *
 * Out of bounds values wrap around.
 *
 * @param nMasterHandle  master board handle
 * @param x              pixel x value, 0-256
 * @param y              pixel row
 * @param val            pixel pair value, the upper nibble being the leftmost
*/
MAGRESWCH_API void DioWriteOLEDPixelImm(dio_handle_h nMasterHandle, uint8_t row, uint8_t column, uint8_t val);

/**
 * @brief Writes a pixel to the screen at the designated coordinate in the local
 * screen buffer.
 *
 * Out of bounds values wrap around.
 *
 * @param nMasterHandle  master board handle
 * @param mode           buffer overwrite mode
 * @param x              pixel x value, 0-256
 * @param y              pixel row
 * @param val            pixel pair value, the upper nibble being the leftmost
*/
MAGRESWCH_API void DioWriteOLEDPixelBuff(dio_handle_h nMasterHandle, dio_oled_write_mode_e mode,
    uint8_t x, uint8_t y, uint8_t val);

/**
 * @brief Immediately writes a line to the screen that begins at (x1, y1), with value `val1`,
 * to (x2, y2) with value `val2`.
 *
 * `val1` and `val2` are linearly interpolated.
 *
 * Undefined behavior if coordinates are outside bounds of display
 *
 * @param nMasterHandle  master board handle
 * @param x1             starting x value
 * @param y1             starting y value
 * @param x2             ending x value
 * @param y2             ending y value
 * @param val1           starting brightness value
 * @param val2           ending brightness value
*/
MAGRESWCH_API void DioWriteOLEDLineImm(dio_handle_h nMasterHandle,
    uint8_t x1,
    uint8_t y1,
    uint8_t x2,
    uint8_t y2,
    uint8_t val1,
    uint8_t val2);


/**
 * @brief Writes a line to the local screen buffer that begins at (x1, y1), with
 * value `val1`, to (x2, y2) with value `val2`.
 *
 * `val1` and `val2` are linearly interpolated.
 *
 * Undefined behavior if coordinates are outside bounds of display
 *
 * @param nMasterHandle  master board handle
 * @param mode           buffer overwrite mode
 * @param x1             starting x value
 * @param y1             starting y value
 * @param x2             ending x value
 * @param y2             ending y value
 * @param val1           starting brightness value
 * @param val2           ending brightness value
*/
MAGRESWCH_API void DioWriteOLEDLineBuff(dio_handle_h nMasterHandle,
    dio_oled_write_mode_e mode,
    uint8_t x1,
    uint8_t y1,
    uint8_t x2,
    uint8_t y2,
    uint8_t val1,
    uint8_t val2);

/**
 * @brief Writes an array of length len to screen immediately.
 *
 * The array is written to the screen at a y offset and single pixel offset in
 * the x direction, nibble calculations to move the array across by one pixel
 * are done internally.
 *
 * Undefined behavior if coordinates are outside bounds of display
 *
 * @param nMasterHandle  master board handle
 * @param x              x offset
 * @param y              y offset
 * @param arr            pointer to the array
 * @param len            length of the array
*/
MAGRESWCH_API void DioWriteOLEDArrImm(dio_handle_h nMasterHandle, uint8_t x, uint8_t y, uint8_t *arr, size_t len);


/**
 * @brief Writes an array of length len to the local screen buffer.
 *
 * The array is written to the screen at a y offset and single pixel offset in
 * the x direction, nibble calculations to move the array across by one pixel
 * are done internally.
 *
 * Undefined behavior if coordinates are outside bounds of display
 *
 * @param nMasterHandle  master board handle
 * @param mode           buffer overwrite mode
 * @param x              x offset
 * @param y              y offset
 * @param arr            pointer to the array
 * @param len            length of the array
*/
MAGRESWCH_API void DioWriteOLEDArrBuff(dio_handle_h nMasterHandle, dio_oled_write_mode_e mode,
    uint8_t x, uint8_t y, uint8_t *arr, size_t len);

/**
 * @brief Writes 128x64 bytes to the screen at once, given a single dimensional
 * array of that size.
 *
 * Each byte represents two pixels, the upper nibble representing the leftmost
 * pixel of the byte. The array MUST be that size, otherwise program will
 * segfault.
 *
 * @param nMasterHandle  master board handle
 * @param arr      array to write
 */
MAGRESWCH_API void DioWriteOLEDScreenArrImm(dio_handle_h nMasterHandle, uint8_t *arr);

/**
 * @brief Writes 128x64 bytes to the local screen buffer, given a single dimensional
 * array of that size.
 *
 * Each byte represents two pixels, the upper nibble representing the leftmost
 * pixel of the byte. The array MUST be that size, otherwise program will
 * segfault.
 *
 * @param nMasterHandle  master board handle
 * @param mode     buffer overwrite mode
 * @param arr      array to write
 */
MAGRESWCH_API void DioWriteOLEDScreenArrBuff(dio_handle_h nMasterHandle, dio_oled_write_mode_e
    mode, uint8_t *arr);


/**
 * @brief Writes an array of length 128 to a row on screen immediately.
 *
 * The array is written to the screen at a y offset.
 *
 * @param nMasterHandle  master board handle
 * @param y              y offset
 * @param arr            pointer to the array
*/
MAGRESWCH_API void DioWriteOLEDRowArrImm(dio_handle_h nMasterHandle, uint8_t y, uint8_t *arr);


/**
 * @brief Writes an array of length 128 to a row in the local screen buffer.
 *
 * The array is written to the screen at a y offset.
 *
 * @param nMasterHandle  master board handle
 * @param y              y offset
 * @param arr            pointer to the array
*/
MAGRESWCH_API void DioWriteOLEDRowArrBuff(dio_handle_h nMasterHandle, dio_oled_write_mode_e mode,
    uint8_t y, uint8_t *arr);

/**
 * @brief Writes singular value to a row on screen immediately.
 *
 * The value is written to the screen at a y offset.
 *
 * @param nMasterHandle  master board handle
 * @param y              y offset
 * @param val            value to write
*/
MAGRESWCH_API void DioWriteOLEDRowValImm(dio_handle_h nMasterHandle, uint8_t y, uint8_t val);

/**
 * @brief Writes an array of length `col_width*height` to the screen immediately.
 *
 * The array is one dimensional, but written to the screen two-dimensionally at
 * a y offset and single pixel offset in the x direction. The array is split
 * into strips of `col_width`, and is written in `height` rows. The array should be
 * formatted such that the first byte is the top-left of the image, and the last
 * byte is the bottom right.
 *
 * Undefined behavior if coordinates are outside bounds of display
 *
 * @param nMasterHandle  master board handle
 * @param mode           buffer overwrite mode
 * @param x              x offset
 * @param y              y offset
 * @param col_width      width of image to write
 * @param height         height of image to write
 * @param arr            pointer to the array
*/
MAGRESWCH_API void DioWriteOLEDAreaArrImm(dio_handle_h nMasterHandle, uint8_t x, uint8_t y,
    uint8_t col_width, uint8_t height, uint8_t *arr);

/**
 * @brief Writes an array of length `col_width*height` to the local screen buffer.
 *
 * The array is one dimensional, but written to the screen two-dimensionally at
 * a y offset and single pixel offset in the x direction. The array is split
 * into strips of `col_width`, and is written in `height` rows. The array should be
 * formatted such that the first byte is the top-left of the image, and the last
 * byte is the bottom right.
 *
 * Undefined behavior if coordinates are outside bounds of display
 *
 * @param nMasterHandle  master board handle
 * @param mode           buffer overwrite mode
 * @param x              x offset
 * @param y              y offset
 * @param col_width      width of image to write
 * @param height         height of image to write
 * @param arr            pointer to the array
*/
MAGRESWCH_API void DioWriteOLEDAreaArrBuff(dio_handle_h nMasterHandle, dio_oled_write_mode_e
    mode, uint8_t x, uint8_t y,
    uint8_t col_width, uint8_t height,
    uint8_t *arr);

/**
 * @brief Writes 128x64 bytes to the screen at once, given a single dimensional
 * array of that size.
 *
 * The array MUST be that size, otherwise program will segfault.
 *
 * @param nMasterHandle  master board handle
 * @param arr      array to write
 */
MAGRESWCH_API void DioWriteOLED2DUnpackedArrImm(dio_handle_h nMasterHandle, uint8_t **arr);

/**
 * @brief Sends the current local screen buffer to the OLED screen.
 *
 * @param nMasterHandle  master board handle
 */
MAGRESWCH_API void DioSendOLEDBuff(dio_handle_h nMasterHandle);

/**
 * @brief Clears the current local screen buffer.
 *
 * @param nMasterHandle  master board handle
 */
MAGRESWCH_API void DioClearOLEDBuff(dio_handle_h nMasterHandle);

/**
 * @brief Clears the screen immediately.
 *
 * @param nMasterHandle  master board handle
 */
MAGRESWCH_API void DIOClearOLEDImm(dio_handle_h nMasterHandle);

/**
 * @brief Opens and allocates memory for a bitmap file.
 *
 * @param nMasterHandle  master board handle
 * @param path           null terminated path to the file
 * @param lpnStatus       returned status, 0 on success, < 0 failure.
 * @retval               returns a handle to the bmp data
 */
MAGRESWCH_API dio_bmp_h DioOpenBMP(dio_handle_h nMasterHandle, char *path, int16_t *lpnStatus);


/**
 * @brief Closes and deallocates memory for a bitmap file.
 *
 * @param nMasterHandle  master board handle
 * @param bmp_handle     handle of the bmp file
 */
MAGRESWCH_API void DioCloseBMP(dio_handle_h nMasterHandle,
    dio_bmp_h *bmp_handle);


/**
 * @brief Writes a BMP to the screen immediately at an x and y offset.
 *
 * The origin of the BMP is the top left corner, and the offset is the distance
 * between this corner and the BMP origin.
 *
 * @param nMasterHandle  master board handle
 * @param x              x offset
 * @param y              y offset
 * @param bmp_handle     handle of the bmp file
*/
MAGRESWCH_API void DioWriteOLEDAreaBMPImm(dio_handle_h nMasterHandle, uint8_t x, uint8_t y,
    dio_bmp_h bmp_handle);

/**
 * @brief Writes a BMP to the local screen buffer at an x and y offset.
 *
 * The origin of the BMP is the top left corner, and the offset is the distance
 * between this corner and the BMP origin.
 *
 * @param nMasterHandle  master board handle
 * @param mode           buffer overwrite mode
 * @param x              x offset
 * @param y              y offset
 * @param bmp_handle     handle of the bmp file
*/
MAGRESWCH_API void DioWriteOLEDAreaBMPBuff(dio_handle_h nMasterHandle,
    dio_oled_write_mode_e mode, uint8_t x, uint8_t y, dio_bmp_h bmp_handle);

/**
 * @brief Writes a single character immediately to the screen.
 *
 * @param nMasterHandle  master board handle
 * @param x              x offset
 * @param y              y offset
 * @param letter         letter to write
*/
MAGRESWCH_API void DioWriteOLEDCharImm(dio_handle_h nMasterHandle, uint8_t x, uint8_t y, char letter);


/**
 * @brief Writes a single character to the local screen buffer.
 *
 * @param nMasterHandle  master board handle
 * @param x              x offset
 * @param y              y offset
 * @param letter         letter to write
*/
MAGRESWCH_API void DioWriteOLEDCharBuff(dio_handle_h nMasterHandle,
    dio_oled_write_mode_e mode,
    uint8_t x, uint8_t y,
    char letter);


/**
 * @brief Writes a string immediately to the screen.
 *
 * @param nMasterHandle  master board handle
 * @param x              x offset
 * @param y              y offset
 * @param str            string to write
*/
MAGRESWCH_API void DioWriteOLEDStrImm(dio_handle_h nMasterHandle,
    uint8_t x, uint8_t y,
    char* str);

/**
 * @brief Writes a string to the local screen buffer.
 *
 * @param nMasterHandle  master board handle
 * @param x              x offset
 * @param y              y offset
 * @param str            string to write
*/
MAGRESWCH_API void DioWriteOLEDStrBuff(dio_handle_h nMasterHandle,
    dio_oled_write_mode_e mode,
    uint8_t x, uint8_t y,
    char* str);

/**
 * @brief Returns a null-terminated string associated with an `lpnStatus` error
 *
 * @param nMasterHandle  master board handle
 * @param lpnStatus      lpnStatus, assumed to be in error
 * @retval               returns null terminated error string
*/
MAGRESWCH_API const char* DioGetErrorStr(dio_handle_h nMasterHandle, int16_t lpnStatus);

#ifdef __cplusplus
}
#endif
